(*
 * Copyright (C) Citrix Systems Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; version 2.1 only. with the special
 * exception on linking described in file LICENSE.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *)

open Xcp_service
open Common

let driver = "ffs"
let name = "ffs"
let description = "Flat File Storage Repository for XCP"
let vendor = "Citrix"
let copyright = "Citrix Inc"
let required_api_version = "2.0"

let supported_formats = [ Vhd; Raw; Qcow2 ]

let features = [
  "VDI_CREATE";
  "VDI_DELETE";
  "VDI_ATTACH";
  "VDI_DETACH";
  "VDI_ACTIVATE";
  "VDI_DEACTIVATE";
  "VDI_SNAPSHOT";
  "VDI_CLONE";
  "VDI_RESIZE"
] @ (List.map (fun x -> Printf.sprintf "FORMAT_%s" (String.uppercase (string_of_format x))) supported_formats)
let _path = "path"
let _location = "location"
let _format = "format"
let configuration = [
   _path, "path to store images and metadata";
   _location, "remote path to store images and metadata (use server:path)";
   _format, Printf.sprintf "default format for disks (one of %s)" (String.concat ", " (List.map string_of_format supported_formats));
]
let _type = "type" (* in sm-config *)

let state_root = ref "/var/run/nonpersistent"
let state_path () = Printf.sprintf "%s/%s.%s" !state_root name json_ext
let mount_path = ref "/var/run/sr-mount"
let device_ext = "device"

let dot_regexp = Re_str.regexp_string "."
let extension x = match Re_str.split_delim dot_regexp x with
  | [] -> ""
  | x -> List.hd (List.rev x)

let colon_regexp = Re_str.regexp_string ":"

type srs = (string * sr) list with rpc

open Storage_interface

let default_format = ref Vhd

let format_of_kvpairs key default x =
  match (if List.mem_assoc key x
    then format_of_string (List.assoc key x)
    else None) with
  | Some x -> x
  | None -> default

let set_default_format x =
  begin match (format_of_string x) with
    | Some x ->
      default_format := x;
    | None ->
      ()
  end;
  info "Default disk format will be: %s" (string_of_format !default_format)

let get_default_format () = string_of_format !default_format

module Attached_srs = struct
  let table = Hashtbl.create 16
  let save () =
    let srs = Hashtbl.fold (fun id sr acc -> (id, sr) :: acc) table [] in
    let txt = Jsonrpc.to_string (rpc_of_srs srs) in
    let dir = Filename.dirname (state_path ()) in
    if not(Sys.file_exists dir)
    then mkdir_rec dir 0o0755;
    file_of_string (state_path ()) txt
  let load () =
    let state_path = state_path () in
    if Sys.file_exists state_path then begin
      info "Loading state from: %s" state_path;
      let all = string_of_file state_path in
      let srs = srs_of_rpc (Jsonrpc.of_string all) in
      Hashtbl.clear table;
      List.iter (fun (id, sr) -> Hashtbl.replace table id sr) srs
    end else info "No saved state; starting with an empty configuration"

  (* On service start, load any existing database *)
  let _ = load ()
  let get id =
    if not(Hashtbl.mem table id)
    then raise (Sr_not_attached id)
    else Hashtbl.find table id
  let put id sr =
    (* We won't fail if the SR already attached. FIXME What if the user attempts
       to attach us twice with different configuration? *)
    Hashtbl.replace table id sr;
    save ()
  let remove id =
    Hashtbl.remove table id
end

module Implementation = struct
  type context = unit

  module Query = struct
    let query ctx ~dbg = {
        driver;
        name;
        description;
        vendor;
        copyright;
        version = Version.version;
        required_api_version;
        features;
        configuration;
    }

    let diagnostics ctx ~dbg = "Not available"
  end
  module DP = struct include Storage_skeleton.DP end
  module VDI = struct
    (* The following are all not implemented: *)
    include Storage_skeleton.VDI

    let device_path_of sr vdi = Printf.sprintf "/var/run/nonpersistent/%s/%s/%s.%s" name sr.sr vdi device_ext

    let md_path_of sr vdi =
        vdi_path_of sr vdi ^ "." ^ json_ext

    let vdi_info_of_path path =
        let md_path = path ^ "." ^ json_ext in
        let readme_path = path ^ "." ^ readme_ext in
        if Sys.file_exists md_path then begin
          let txt = string_of_file md_path in
          Some (vdi_info_of_rpc (Jsonrpc.of_string txt))
        end else begin
          let open Unix.LargeFile in
          let stats = stat path in
          let ext = extension path in
          (* We usually can't store additonal data in read/only directories
             full of .iso images. We assume these files are Raw.
             Openstack wants to create files with extension .vhd and have
             these detected as vhds. *)
          let ext_format = [
            iso_ext, Raw;
            vhd_ext, Vhd;
            qcow2_ext, Qcow2;
          ] in

          (* We hide any .vhd which has a parent metadata file *)
          if stats.st_kind <> Unix.S_REG
          || ext = json_ext
          || ext = readme_ext
          || (Sys.file_exists readme_path)
          then None
          else Some {
            vdi = Filename.basename path;
            content_id = "";
            name_label = Filename.basename path;
            name_description = "";
            ty = "user";
            metadata_of_pool = "";
            is_a_snapshot = false;
            snapshot_time = iso8601_of_float 0.;
            snapshot_of = "";
            read_only = false;
            virtual_size =
              if ext = vhd_ext
              then Vhdformat.get_virtual_size path
              else if ext = qcow2_ext
              then Qemu.get_virtual_size path
              else stats.st_size;
            physical_utilisation = stats.st_size;
            sm_config =
              if List.mem_assoc ext ext_format 
              then [ _type, string_of_format (List.assoc ext ext_format) ]
              else [];
            persistent = true;
          }
        end

   let vdi_format_of sr vdi =
     match vdi_info_of_path (vdi_path_of sr vdi) with
     | None ->
       error "VDI %s/%s has no associated vdi_info - I don't know how to operate on it." sr.sr vdi;
       failwith (Printf.sprintf "VDI %s/%s has no vdi_info" sr.sr vdi)
     | Some vdi_info ->
       begin
         if not(List.mem_assoc _type vdi_info.sm_config) then begin
           error "VDI %s/%s has no sm_config:type - I don't know how to operate on it." sr.sr vdi;
           failwith (Printf.sprintf "VDI %s/%s has no sm-config:type" sr.sr vdi)
         end;
         let t = List.assoc _type vdi_info.sm_config in
         match format_of_string t with
         | Some x -> x
         | None ->
           error "VDI %s/%s has an unrecognised sm_config:type=%s - I don't know how to operate on it." sr.sr vdi t;
           failwith (Printf.sprintf "VDI %s/%s has unrecognised sm-config:type=%s" sr.sr vdi t)
       end

    let choose_filename sr vdi_info =
      let existing = Sys.readdir sr.path |> Array.to_list in
      let name_label =
        (* empty filenames are not valid *)
        if vdi_info.name_label = ""
        then "unknown"
        else
          (* only some characters are valid in filenames *)
          let name_label = String.copy vdi_info.name_label in
          for i = 0 to String.length name_label - 1 do
            name_label.[i] <- match name_label.[i] with
              | 'a' .. 'z'
              | 'A' .. 'Z'
              | '0' .. '9'
              | '-' | '_' | '+' -> name_label.[i]
              | _ -> '_'
          done;
          name_label in
      if not(List.mem name_label existing)
      then name_label
      else
        let stem = name_label ^ "." in
        let with_common_prefix = List.filter (startswith stem) existing in
        let suffixes = List.map (remove_prefix stem) with_common_prefix in
        let highest_number = List.fold_left (fun acc suffix ->
          let this = try int_of_string suffix with _ -> 0 in
          max acc this) 0 suffixes in
        stem ^ (string_of_int (highest_number + 1))

    let create ctx ~dbg ~sr ~vdi_info =
      let sr = Attached_srs.get sr in
      let format = format_of_kvpairs _type sr.format vdi_info.sm_config in
      let sm_config = (_type, string_of_format format) :: (List.filter (fun (k, _) -> k <> _type) vdi_info.sm_config) in
      let vdi_info = { vdi_info with
        vdi = choose_filename sr vdi_info;
        snapshot_time = iso8601_of_float 0.;
        sm_config;
      } in
      let vdi_path = vdi_path_of sr vdi_info.vdi in
      let md_path = md_path_of sr vdi_info.vdi in

      begin match format with
      | Vhd -> Vhdformat.create vdi_path vdi_info.virtual_size
      | Raw -> Sparse.create vdi_path vdi_info.virtual_size
      | Qcow2 -> Qemu.create vdi_path vdi_info.virtual_size
      end;
      debug "VDI.create %s -> %s (%Ld)" vdi_info.name_label vdi_path vdi_info.virtual_size;  
      file_of_string md_path (Jsonrpc.to_string (rpc_of_vdi_info vdi_info));
      vdi_info

    let destroy ctx ~dbg ~sr ~vdi =
      let sr = Attached_srs.get sr in
      let vdi_path = vdi_path_of sr vdi in
      if not(Sys.file_exists vdi_path) && not(Sys.file_exists (md_path_of sr vdi))
      then raise (Vdi_does_not_exist vdi);

      debug "VDI.destroy %s" vdi;
      let format = vdi_format_of sr vdi in
      begin match format with
      | Vhd | Qcow2 -> Disk_tree.rm format sr vdi
      | Raw -> Sparse.destroy vdi_path
      end;

      rm_f (md_path_of sr vdi)

    let snapshot_clone_common is_a_snapshot ctx ~dbg ~sr ~vdi_info =
      let sr = Attached_srs.get sr in
      let vdi = vdi_info.vdi in
      let vdi_path = vdi_path_of sr vdi in
      let md_path = md_path_of sr vdi in
      if not(Sys.file_exists vdi_path) && not(Sys.file_exists md_path)
      then raise (Vdi_does_not_exist vdi);
      let format = vdi_format_of sr vdi in
      info "VDI.clone %s (format = %s)" vdi (string_of_format format);

      let parent_vdi_info = match vdi_info_of_path vdi_path with
      | None -> raise (Vdi_does_not_exist vdi)
      | Some info -> info in

      let snapshot_fn, leaf_type = match format with
      | Vhd | Raw ->
        Vhdformat.snapshot, Vhd
      | Qcow2 ->
        Qemu.snapshot, Qcow2 in

      let base = choose_filename sr vdi_info in
      (* TODO: stop renaming because it causes problems on NFS *)
      info "rename %s -> %s" vdi_path (vdi_path_of sr base);
      Disk_tree.rename format sr vdi base;
      Unix.rename vdi_path (vdi_path_of sr base);
      snapshot_fn vdi_path (vdi_path_of sr base) format parent_vdi_info.virtual_size;
      let snapshot = choose_filename sr vdi_info in
      snapshot_fn (vdi_path_of sr snapshot) (vdi_path_of sr base) format parent_vdi_info.virtual_size;
      Disk_tree.(write sr base { children = [ vdi; snapshot ] });
      let vdi_info = { vdi_info with
        vdi = snapshot;
        virtual_size = parent_vdi_info.virtual_size;
        physical_utilisation = 0L;
        is_a_snapshot;
        snapshot_of = if is_a_snapshot then vdi else "";
        (* TODO: snapshot_time *)
        sm_config = [ _type, string_of_format leaf_type ];
      } in
      file_of_string (md_path_of sr snapshot) (Jsonrpc.to_string (rpc_of_vdi_info vdi_info));
      vdi_info

    let snapshot = snapshot_clone_common true
    let clone = snapshot_clone_common false

    let modify_vdi_info sr vdi f =
      let sr = Attached_srs.get sr in
      let vdi_path = vdi_path_of sr vdi in
      let md_path = md_path_of sr vdi in
      match vdi_info_of_path vdi_path with
      | None ->
        raise (Vdi_does_not_exist vdi);
      | Some vdi_info ->
        let vdi_info = f vdi_info in
        file_of_string md_path (Jsonrpc.to_string (rpc_of_vdi_info vdi_info))

    let add_to_sm_config ctx ~dbg ~sr ~vdi ~key ~value =
      info "VDI.add_to_sm_config dbg:%s sr:%s vdi:%s key:%s value:%s" dbg sr vdi key value;
      modify_vdi_info sr vdi
        (fun vdi_info ->
          let sm_config = List.filter (fun (k, _) -> k <> key) vdi_info.sm_config in
          { vdi_info with sm_config = (key, value) :: sm_config }
        )

    let remove_from_sm_config ctx ~dbg ~sr ~vdi ~key =
      info "VDI.remove_from_sm_config dbg:%s sr:%s vdi:%s key:%s" dbg sr vdi key;
      modify_vdi_info sr vdi
        (fun vdi_info ->
          let sm_config = List.filter (fun (k, _) -> k <> key) vdi_info.sm_config in
          { vdi_info with sm_config }
        )

    let set_content_id ctx ~dbg ~sr ~vdi ~content_id =
      info "VDI.set_content_id dbg:%s sr:%s vdi:%s content_id:%s" dbg sr vdi content_id;
      modify_vdi_info sr vdi (fun vdi_info -> { vdi_info with content_id })

    let similar_content ctx ~dbg ~sr ~vdi =
      info "VDI.similar_content dbg:%s sr:%s vdi:%s" dbg sr vdi;
      []

    let compose ctx ~dbg ~sr ~vdi1 ~vdi2 =
      info "VDI.compose dbg:%s sr:%s vdi1:%s vdi2:%s" dbg sr vdi1 vdi2;
      let sr = Attached_srs.get sr in

      let format1 = vdi_format_of sr vdi1 in
      let format2 = vdi_format_of sr vdi2 in
      if format1 <> format2
      then failwith "VDI.compose can't mix formats";
      if format1 <> Vhd
      then failwith "VDI.compose only supports vhd";

      let vdi1_path = vdi_path_of sr vdi1 in
      let vdi2_path = vdi_path_of sr vdi2 in
      let old_parent = match Vhdformat.get_parent vdi2_path with
        | None ->
          error "VDI.compose dbg:%s %s has no parent; is not a differencing disk" dbg vdi2_path;
          failwith "VDI is not a differencing disk"
        | Some x -> x in

      info "VDI.compose dbg:%s %s/parent <- %s" dbg vdi2_path vdi1_path;
      Vhdformat.set_parent vdi2_path vdi1_path;

      let children_of vdi = match Disk_tree.read sr vdi with
        | None -> []
        | Some c -> c.Disk_tree.children in

      (* Update the readme of vdi2's old parent: remove vdi2 *)
      info "VDI.compose dbg:%s %s/children = [ %s ] - %s" dbg old_parent (String.concat ", " (children_of old_parent)) vdi2;
      Disk_tree.(write sr old_parent { children = List.filter (fun x -> x <> vdi2) (children_of old_parent) });

      (* Update the readme of vdi2's new parent: add vdi2 *)
      info "VDI.compose dbg:%s %s/children = [ %s ] + %s" dbg vdi1 (String.concat ", " (children_of vdi1)) vdi2;
      Disk_tree.(write sr vdi1 { children = vdi2 :: (children_of vdi1) });

      (* Signal tapdisk that the chain has changed *)
      let symlink = device_path_of sr vdi2 in
      let device = Unix.readlink symlink in
      info "VDI.compose dbg:%s refreshing %s" dbg device;
      Vhdformat.refresh device

    let resize ctx ~dbg ~sr ~vdi ~new_size =
      info "VDI.resize %s %s %Ld" sr vdi new_size;
      let sr = Attached_srs.get sr in
      let vdi_path = vdi_path_of sr vdi in
      let md_path = md_path_of sr vdi in
      match vdi_info_of_path vdi_path with
      | None ->
        raise (Vdi_does_not_exist vdi);
      | Some vdi_info ->
        let resize_fn = match vdi_format_of sr vdi with
        | Raw -> fun _ _ -> raise (Unimplemented "raw resize not currently implemented")
        | Vhd -> Vhdformat.resize
        | Qcow2 -> fun path new_size -> Qemu.resize path new_size in
        let new_size = resize_fn vdi_path new_size in
        let vdi_info = { vdi_info with virtual_size = new_size } in
        file_of_string md_path (Jsonrpc.to_string (rpc_of_vdi_info vdi_info));
        new_size

    let stat ctx ~dbg ~sr ~vdi =
      let sr = Attached_srs.get sr in
      let md_path = md_path_of sr vdi in
      vdi_info_of_rpc (Jsonrpc.of_string (string_of_file md_path))

    let attach ctx ~dbg ~dp ~sr ~vdi ~read_write =
      let sr = Attached_srs.get sr in
      let vdi_path = vdi_path_of sr vdi in
      let attach_info = match vdi_format_of sr vdi with
      | Vhd -> Vhdformat.attach vdi_path read_write
      | Raw -> Sparse.attach vdi_path read_write
      | Qcow2 -> Qemu.attach vdi_path read_write
      in
      let symlink = device_path_of sr vdi in
      mkdir_rec (Filename.dirname symlink) 0o700;
      Unix.symlink attach_info.params symlink;
      attach_info

    let detach ctx ~dbg ~dp ~sr ~vdi =
      let sr = Attached_srs.get sr in
      let symlink = device_path_of sr vdi in
      let device = Unix.readlink symlink in
      (* We can get transient failures from background tasks on the system
         inspecting the block device. We must not allow detach to fail, so
         we should keep retrying until the transient failures stop happening. *)
      retry_every 0.1 (fun () ->
        match vdi_format_of sr vdi with
        | Vhd -> Vhdformat.detach device
        | Raw -> Sparse.detach device
        | Qcow2 -> Qemu.detach device
      );
      rm_f symlink
    let activate ctx ~dbg ~dp ~sr ~vdi =
      let sr = Attached_srs.get sr in
      let symlink = device_path_of sr vdi in
      let device = Unix.readlink symlink in
      let path = vdi_path_of sr vdi in
      begin match vdi_format_of sr vdi with
      | Vhd -> Vhdformat.activate device path Tapctl.Vhd
      | Raw -> Sparse.activate device path
      | Qcow2 -> Qemu.activate device path
      end
    let deactivate ctx ~dbg ~dp ~sr ~vdi =
      let sr = Attached_srs.get sr in
      let symlink = device_path_of sr vdi in
      let device = Unix.readlink symlink in
      begin match vdi_format_of sr vdi with
      | Vhd -> Vhdformat.deactivate device
      | Raw -> Sparse.deactivate device
      | Qcow2 -> Qemu.deactivate device
      end
  end
  module SR = struct
    include Storage_skeleton.SR
    let stat ctx ~dbg ~sr =
      let sr = Attached_srs.get sr in
      let x = Statvfs.statvfs sr.path in
      let total_space = Int64.mul x.Statvfs.f_blocks x.Statvfs.f_frsize in
      let free_space = Int64.mul x.Statvfs.f_bavail x.Statvfs.f_bsize in
      { total_space; free_space }

    let scan ctx ~dbg ~sr =
       let sr = Attached_srs.get sr in
       if not(Sys.file_exists sr.path)
       then []
       else
          Sys.readdir sr.path
            |> Array.to_list
            |> List.map (Filename.concat sr.path)
            |> List.map VDI.vdi_info_of_path
            |> List.fold_left (fun acc x -> match x with
               | None -> acc
               | Some x -> x :: acc) []

    let destroy = destroy
    let reset = reset
    let detach ctx ~dbg ~sr =
       begin
         try
           let sr = Attached_srs.get sr in
           if sr.is_mounted then Mount.umount sr.path
         with
         | Sr_not_attached _ -> ()
         | e ->
           error "Failed to umount %s: mountpoint may have leaked" sr;
       end;
       Attached_srs.remove sr
    let attach ctx ~dbg ~sr ~device_config =
       let has_path = List.mem_assoc _path device_config in
       let has_location = List.mem_assoc _location device_config in
       if not has_path && (not has_location) then begin
           error "Required device_config:path not present";
           raise (Missing_configuration_parameter _path);
       end;
       (* Explicitly reject formats that we don't recognise *)
       if List.mem_assoc _format device_config then begin
         let format = List.assoc _format device_config in
         match format_of_string format with
         | None ->
           error "The format '%s' is not supported. Try one of %s." format
             (String.concat ", " (List.map string_of_format supported_formats));
           raise (Unimplemented format)
         | Some _ -> ()
       end;
         
       let format = format_of_kvpairs _format !default_format device_config in
       let path =
         if has_path then List.assoc _path device_config
         else if has_location then List.assoc _location device_config
         else assert false in
       (* If 'path' is of the form server:path, then we perform a mount *)
       match Re_str.bounded_split_delim colon_regexp path 2 with
       | [ _ ] ->
         let is_mounted = false in
         Attached_srs.put sr { sr; path; is_mounted; format }
       | [ _; _ ] ->
         (* mount remote_host *)
         let local_path = Filename.concat !mount_path sr in
         mkdir_rec local_path 0o0755;
         Mount.mount path local_path;
         let is_mounted = true in
         Attached_srs.put sr { sr; path = local_path; is_mounted; format }
       | _ ->
         error "Failed to parse device_config parameter: %s (expected either <path> or <server>:<path>)" path;
         failwith "expected <path> or <server>:<path>" 
    let create ctx ~dbg ~sr ~device_config ~physical_size =
       (* attach will validate the device_config parameters *)
       attach ctx ~dbg ~sr ~device_config;
       detach ctx ~dbg ~sr
  end
  module UPDATES = struct include Storage_skeleton.UPDATES end
  module TASK = struct include Storage_skeleton.TASK end
  module Policy = struct include Storage_skeleton.Policy end
  module DATA = struct include Storage_skeleton.DATA end
  let get_by_name = Storage_skeleton.get_by_name

end

module Server = Storage_interface.Server(Implementation)

