(*
 * Copyright (C) 2013 Citrix Systems Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; version 2.1 only. with the special
 * exception on linking described in file LICENSE.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *)

let project_url = "http://github.com/djs55/ocaml-qmp"

let default_path = ref "/tmp/qmp"

open Common
open Cmdliner

(* Help sections common to all commands *)
let _common_options = "COMMON OPTIONS"
let help = [
  `S _common_options;
  `P "These options are common to all commands.";
  `S "MORE HELP";
  `P "Use `$(mname) $(i,COMMAND) --help' for help on a single command."; `Noblank;
  `S "BUGS"; `P (Printf.sprintf "Check bug reports at %s" project_url);
]

(* Options common to all commands *)
let common_options_t =
  let docs = _common_options in
  let debug =
    let doc = "Give only debug output." in
    Arg.(value & flag & info ["debug"] ~docs ~doc) in
  let verb =
    let doc = "Give verbose output." in
    let verbose = true, Arg.info ["v"; "verbose"] ~docs ~doc in
    Arg.(last & vflag_all [false] [verbose]) in
  let socket =
    let doc = Printf.sprintf "Specify path to the server Unix domain socket." in
    Arg.(value & opt file !default_path & info ["socket"] ~docs ~doc) in
  Term.(pure Common.make $ debug $ verb $ socket)

let default_cmd =
  let doc = "interact with a running qemu via QMP" in
  let man = help in
  Term.(ret (pure (fun _ -> `Help (`Pager, None)) $ common_options_t)),
  Term.info "qmp-cli" ~version:"1.0.0" ~sdocs:_common_options ~doc ~man

let watch_cmd =
  let doc = "watch for asynchronous events" in
  let man = [
    `S "DESCRIPTION";
    `P "Watches for and prints asynchronous events generated by the VM.";
  ] @ help in
  Term.(pure Client.watch $ common_options_t),
  Term.info "watch" ~sdocs:_common_options ~doc ~man

let stop_cmd =
  let doc = "Immediately freeze execution" in
  let man = [
    `S "DESCRIPTION";
    `P "Stop the VM running but leave the state intact. Use 'cont' to start the VM executing again.";
  ] @ help in
  Term.(pure Client.stop $ common_options_t),
  Term.info "stop" ~sdocs:_common_options ~doc ~man

let cont_cmd =
  let doc = "Continue a frozen VM" in
  let man = [
    `S "DESCRIPTION";
    `P "If a VM has been frozen with 'stop', then 'cont' will cause it to start executing again.";
  ] @ help in
  Term.(pure Client.cont $ common_options_t),
  Term.info "cont" ~sdocs:_common_options ~doc ~man

let powerdown_cmd =
  let doc = "Press the system powerdown button" in
  let man = [
    `S "DESCRIPTION";
    `P "Press the system powerdown button which requests that the OS shuts itself down cleanly.";
  ] @ help in
  Term.(pure Client.system_powerdown $ common_options_t),
  Term.info "powerdown" ~sdocs:_common_options ~doc ~man

let cmds = [ watch_cmd; stop_cmd; cont_cmd; powerdown_cmd ]

let _ =
  match Term.eval_choice default_cmd cmds with
  | `Error _ -> exit 1
  | _ -> exit 0
